<?php

namespace App\Http\Controllers;

use App\Mail\CustomEmail;
use App\Models\EmailLog;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\RateLimiter;

class EmailController extends Controller
{
    public function showForm()
    {
        $users = User::pluck('email', 'id');
        return view('emails.form', compact('users'));
    }

    public function send(Request $request)
    {
        $request->validate([
            'recipients' => 'required|array',
            'recipients.*' => 'email',
            'external_recipients' => 'nullable|string',
            'subject' => 'required|string|max:255',
            'body' => 'required|string',
            'bcc' => 'nullable|string',
            'attachments.*' => 'file|mimes:pdf,doc,docx,xls,xlsx,png,jpg,jpeg|max:10240', // 10MB max
        ]);

        $recipients = $request->recipients;
        $externalRecipients = array_filter(array_map('trim', explode(',', $request->external_recipients ?? '')));
        $allRecipients = array_unique(array_merge($recipients, $externalRecipients));
        $bcc = array_filter(array_map('trim', explode(',', $request->bcc ?? '')));
        $attachments = [];

        // Handle file uploads
        if ($request->hasFile('attachments')) {
            foreach ($request->file('attachments') as $file) {
                $path = $file->store('attachments', 'public');
                $attachments[] = [
                    'path' => storage_path('app/public/' . $path),
                    'name' => $file->getClientOriginalName(),
                    'mime' => $file->getMimeType(),
                ];
            }
        }

        // Rate limiting: max 100 emails per minute
        $key = 'email-sending:' . auth()->id();
        if (RateLimiter::tooManyAttempts($key, 100)) {
            return back()->withErrors(['error' => 'Too many emails sent. Please try again later.']);
        }

        RateLimiter::hit($key, 60); // 1 minute

        // Send emails individually
        foreach ($allRecipients as $recipient) {
            try {
                // Set timeout for each email
                set_time_limit(30);

                Mail::to($recipient)->send(new CustomEmail(
                    $request->subject,
                    $request->body,
                    $attachments,
                    $bcc
                ));

                // Log success
                EmailLog::create([
                    'recipient_email' => $recipient,
                    'subject' => $request->subject,
                    'body' => $request->body,
                    'status' => 'success',
                    'attachments' => json_encode($attachments),
                ]);
            } catch (\Exception $e) {
                // Log failure
                EmailLog::create([
                    'recipient_email' => $recipient,
                    'subject' => $request->subject,
                    'body' => $request->body,
                    'status' => 'failed',
                    'error_message' => $e->getMessage(),
                    'attachments' => json_encode($attachments),
                ]);

                Log::error("Failed to send email to $recipient: " . $e->getMessage());
            }
        }

        return back()->with('success', 'Emails sent successfully!');
    }

    public function logs()
    {
        $logs = EmailLog::latest()->paginate(10);
        return view('emails.logs', compact('logs'));
    }
}
